#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <netinet/in.h>

#include "myriexpress.h"
#include "mx_raw.h"

#define MXOED_DEBUG 0

#define MAX_PEERS 8192
#define MAX_NICS 8
#define MXOE_PORT 2314
#define MAX_IFC_CNT 16
#define BROADCAST_INTERVAL 1000
#define LONG_BROADCAST_INTERVAL 180000
#define BROADCAST_COUNT 8

/*
 * Info about each NIC
 */
struct mxoed_pkt {
  uint32_t dest_mac_high32;
  uint16_t dest_mac_low16;
  uint16_t src_mac_high16;
  uint32_t src_mac_low32;
  uint16_t proto;		/* ethertype */
  uint16_t sender_peer_index;
  uint8_t pkt_type;
  uint8_t chargap[3];
  uint32_t gap[3];		/* pad to 32 bytes */
  uint32_t nic_id_hi;
  uint32_t nic_id_lo;
  uint32_t serial;
  uint8_t  pad[20];		/* then to 64 bytes */
};

struct nic_info {
  mx_raw_endpoint_t raw_ep;		/* endpoint for this NIC */

  int nic_index;
  uint64_t my_nic_id;
  uint32_t my_serial;

  uint64_t peers[MAX_PEERS];
  uint32_t peer_serial[MAX_PEERS];
  int num_peers;

  int bc_count;
  int bc_interval;

  struct mxoed_pkt outpkt;
  struct mxoed_pkt pkt;
};


/* return difference between two timevals in ms (t1 - t2) */
static inline int32_t
tv_diff(
  struct timeval *t1,
  struct timeval *t2)
{
  int32_t ms;

  ms = (t1->tv_sec - t2->tv_sec) * 1000;
  ms += (t1->tv_usec - t2->tv_usec) / 1000;
  return ms;
}

/*
 * Define this peer to MX
 */
static void
write_peers_to_nic(
  struct nic_info *nip)
{
  mx_return_t rc;
  int i;

  rc = mx_raw_set_route_begin(nip->raw_ep);
  if (rc != MX_SUCCESS) {
    fprintf(stderr, "Error add peer init: %s\n", mx_strerror(rc));
    exit(1);
  }

  /* write self */
  rc = mx_raw_set_route(nip->raw_ep, nip->my_nic_id,
      NULL, 0, 0, 0, MX_HOST_MX);
  if (rc != MX_SUCCESS) {
    fprintf(stderr, "Error adding self to NIC %d: %s\n",
	nip->nic_index, mx_strerror(rc));
    exit(1);
  }
  
  /* add the peers */
  for (i = 0; i < nip->num_peers; i++) {
    rc = mx_raw_set_route(nip->raw_ep, nip->peers[i],
	NULL, 0, 0, 0, MX_HOST_MX);
    if (rc != MX_SUCCESS) {
      fprintf(stderr, "Error adding peer to NIC %d: %s\n",
	  nip->nic_index, mx_strerror(rc));
      exit(1);
    }
  }

  rc = mx_raw_set_route_end(nip->raw_ep);
  if (rc != MX_SUCCESS) {
    fprintf(stderr, "Error add peer init: %s\n", mx_strerror(rc));
    exit(1);
  }

  rc = mx_raw_set_map_version(nip->raw_ep, 0, nip->my_nic_id, 1,
      nip->num_peers+1, 1);
  if (rc != MX_SUCCESS) {
    fprintf(stderr, "Error add peer init: %s\n", mx_strerror(rc));
    exit(1);
  }
}

void
add_peer(
  struct nic_info *nip,
  uint64_t peer_mac,
  int serial)
{

  /* Add this to our local peer table */
  nip->peers[nip->num_peers] = peer_mac;
  nip->peer_serial[nip->num_peers] = serial;
  ++nip->num_peers;

  write_peers_to_nic(nip);
}

int
get_peer_index(
  struct nic_info *nip,
  uint64_t peer_mac)
{
  int i;

  for (i=0; i<nip->num_peers; ++i) {
    if (nip->peers[i] == peer_mac) {
      return i;
    }
  }
  return -1;
}

void
broadcast_my_id(
  struct nic_info *nip)
{
  mx_return_t mxrc;

  mxrc = mx_raw_send(nip->raw_ep, 0, NULL, 0,
      &nip->outpkt, sizeof(nip->outpkt), NULL);
  if (mxrc != MX_SUCCESS) {
    fprintf(stderr, "Error sending raw packet: %s\n", mx_strerror(mxrc));
    exit(1);
  }
#if MXOED_DEBUG
    printf("sent my ID\n");
#endif
}

int
check_for_packet(
  struct nic_info *nip,
  int *elapsed)
{
  mx_return_t mxrc;
  mx_raw_status_t stat;
  void *context;
  uint32_t port;
  uint32_t len;
  struct timeval before;
  struct timeval after;
  int timeout;
  int rc;

  (void) gettimeofday(&before, NULL);

  len = sizeof(nip->pkt);
  if (nip->bc_interval > 0) {
    timeout = nip->bc_interval;
  } else {
    timeout = 0;
  }
  mxrc = mx_raw_next_event(nip->raw_ep, &port, &context,
      &nip->pkt, &len, timeout, &stat);
  if (mxrc != MX_SUCCESS) {
    fprintf(stderr, "Error from mx_raw_next_event: %s\n", mx_strerror(mxrc));
    exit(1);
  }

  (void) gettimeofday(&after, NULL);

  if (stat == MX_RAW_RECV_COMPLETE) {
#if MXOED_DEBUG
    int i;
    unsigned char *p = (unsigned char *)(&nip->pkt);

    printf("recv len = %d\n", len);
    for (i=0; i<16; ++i) printf(" %02x", p[i]); printf("\n");
    for (; i<32; ++i) printf(" %02x", p[i]); printf("\n");
    for (; i<48; ++i) printf(" %02x", p[i]); printf("\n");
#endif

    rc = 1;

  } else if (stat == MX_RAW_SEND_COMPLETE) {
#if MXOED_DEBUG
  printf("send complete\n");
#endif
    rc = 0;

  } else {
    rc = 0;
  }

  /* return elapsed time in ms */
  *elapsed = tv_diff(&after, &before);

#if MXOED_DEBUG
  printf("elapsed = %d\n", *elapsed);
#endif

  return rc;
}

void
process_pkt(
  struct nic_info *nip)
{
  uint32_t nic_half;
  uint64_t nic_id;
  int serial;
  int index;

  /* get peer NIC id from packet */
  nic_id = ntohl(nip->pkt.nic_id_lo);
  nic_half = ntohl(nip->pkt.nic_id_hi);
  nic_id |= ((uint64_t)nic_half << 32);

  serial = ntohl(nip->pkt.serial);

  if (nic_id == nip->my_nic_id) {
    return;
  }

#if MXOED_DEBUG
  printf("got pkt from nic_id %02x%04x, sn=%d ",
      nic_half, ntohl(nip->pkt.nic_id_lo), serial);
#endif

  index = get_peer_index(nip, nic_id);
  if (index == -1) {
#if MXOED_DEBUG
  printf("new peer\n");
#endif
    add_peer(nip, nic_id, serial);
    nip->bc_count = BROADCAST_COUNT;
    nip->bc_interval = 0;
  } else {

    /* new serial number means he likely does not know me */
    if (nip->peer_serial[index] != serial) {
#if MXOED_DEBUG
      printf("known, but serial changed\n");
#endif

      /* record new serial # and broadcast my ID */
      nip->peer_serial[index] = serial;
      nip->bc_count = BROADCAST_COUNT;
      nip->bc_interval = 0;
    } else {
#if MXOED_DEBUG
      printf("already known\n");
#endif
    }
  }
  return;
}

#if MX_OS_MACOSX
#include <sys/param.h>
#define MAX_COMPUTERNAME_LENGTH (MAXHOSTNAMELEN - 1)
#define DWORD size_t
#endif

#if MX_OS_WINNT || MX_OS_MACOSX
void
set_nic_hostname(struct nic_info *nip)
{
  char name[MX_MAX_STR_LEN+1];
  char name2[MAX_COMPUTERNAME_LENGTH+1];
  DWORD name2_len;

  if (mx_nic_id_to_hostname(nip->my_nic_id, name) == MX_SUCCESS) {
    if (MAX_COMPUTERNAME_LENGTH > MX_MAX_STR_LEN) {
      name2_len = MX_MAX_STR_LEN;
    } else {
      name2_len = MAX_COMPUTERNAME_LENGTH;
    }
    name2_len -= 3; /* to allow for :99 mx boards */
    if ((strncmp(name, "localhost", strlen("localhost")) == 0)
#if MX_OS_WINNT
	&& GetComputerName(name2, &name2_len)
#else
	&& (gethostname(name2, name2_len - 1) == 0)
#endif
	) {
      snprintf(name, sizeof(name), "%s:%d", name2, nip->nic_index);
      mx_raw_set_hostname(nip->raw_ep, name);
#if MXOED_DEBUG
      printf("[%s]\n", name);
#endif
    }
  }
}
#endif

void
fill_nic_info(
  struct nic_info *nip)
{
  mx_return_t mxrc;
  uint32_t nic_half;

  mxrc = mx_board_number_to_nic_id(nip->nic_index, &nip->my_nic_id);
  if (mxrc != MX_SUCCESS) {
    fprintf(stderr, "Error getting nic_id for NIC %d\n", nip->nic_index);
    exit(1);
  }

  memset(&nip->outpkt, 0, sizeof(nip->outpkt));
  nip->outpkt.dest_mac_high32 = 0xFFFFFFFF;
  nip->outpkt.dest_mac_low16 = 0xFFFF;
  nip->outpkt.src_mac_high16 = htons((nip->my_nic_id >> 32) & 0xFFFF);
  nip->outpkt.src_mac_low32 = htonl(nip->my_nic_id & 0xFFFFFFFF);
  nip->outpkt.proto = htons(0x86DF);
  nip->outpkt.pkt_type = 1;		

  /* put my nic_id in outbound packet */
  nic_half = (nip->my_nic_id >> 32) & 0xFFFFFFFF;
  nip->outpkt.nic_id_hi = htonl(nic_half);
  nic_half = nip->my_nic_id & 0xFFFFFFFF;
  nip->outpkt.nic_id_lo = htonl(nic_half);

  /* assign a random serial number for this invocation */
  nip->my_serial = random();
  nip->outpkt.serial = htonl(nip->my_serial);
}

void *
nic_thread(
  void *vnip)
{
  struct nic_info *nip;
  int elapsed;
  int rc;

  nip = vnip;

#if MX_OS_WINNT || MX_OS_MACOSX
  set_nic_hostname(nip);
#endif

  fill_nic_info(nip);

  write_peers_to_nic(nip);	/* write self to NIC peer table */

  nip->bc_count = BROADCAST_COUNT;
  nip->bc_interval = 0;

  while (1) {

    /* If broadcasts left to do and interval expired, send one now */
    if (nip->bc_count > 0 && nip->bc_interval <= 0) {
      broadcast_my_id(nip);
      --nip->bc_count;
      if (nip->bc_count > 0) {
	nip->bc_interval = BROADCAST_INTERVAL;
      } else {
	nip->bc_count = 1;
	nip->bc_interval = LONG_BROADCAST_INTERVAL;
      }
    }

    rc = check_for_packet(nip, &elapsed);

    if (rc > 0) {
      process_pkt(nip);
    }

    if (nip->bc_interval > 0) {
      nip->bc_interval -= elapsed;
    }
  }
}


/*
 * Open NICs
 */
void
open_all_nics()
{
  int i;
  int rc;
  int num_nics;
  struct nic_info *nip;
  struct nic_info *nip0;
  mx_raw_endpoint_t ep;
  pthread_t tid;

  num_nics = 0;
  for (i=0; i<MAX_NICS; ++i) {

    /* open raw endpoint for this NIC */
    rc = mx_raw_open_endpoint(i, NULL, 0, &ep);
    if (rc != 0) {
      continue;
    }

    /* allocate NIC info struct */
    nip = (struct nic_info *) calloc(sizeof(*nip), 1);
    if (nip == NULL) {
      fprintf(stderr, "Error allocating NIC info struct\n");
      exit(1);
    }
    nip->raw_ep = ep;
    nip->nic_index = i;

    /* the first NIC will be handled in main thread */
    if (num_nics > 0) {
      rc = pthread_create(&tid, NULL, nic_thread, nip);
      if (rc != 0) {
	fprintf(stderr, "Error creating thread for NIC %d\n", i);
	exit(1);
      }
    } else {
      nip0 = nip;
    }
    ++num_nics;
  }

  /* Error out if no NICs opened */
  if (num_nics == 0) {
    fprintf(stderr, "No NICs found\n");
    exit(1);
  } else {
    nic_thread(nip0);
  }
}
  
int
main(
  int argc,
  char *argv[])
{
  srandom((unsigned int)time(NULL));
  setlinebuf(stdout);
  setlinebuf(stderr);

  /* init mx */
  mx_init();
  open_all_nics();
  exit(0);
}
